﻿namespace Microsoft.Samples.PlanMyNight.Data.Tests
{
    using System;
    using System.Linq;
    using System.Text;
    using Microsoft.Samples.PlanMyNight.Bing;
    using Microsoft.Samples.PlanMyNight.Entities;
    using Microsoft.VisualStudio.TestTools.UnitTesting;
    using Moq;

    [TestClass]
    public class BingActivitiesRepositoryFixture
    {
        [TestInitialize]
        public void InvalidateTokenCache()
        {
            BingActivitiesRepository.InvalidateTokenCache();
        }

        [TestMethod]
        public void RetrieveActivityTypes()
        {
            var repo = new BingActivitiesRepository(new Mock<IBingMapsService>().Object);
            var result = repo.RetrieveActivityTypes();

            Assert.IsNotNull(result);
            Assert.AreEqual(3, result.Count());
        }

        [TestMethod]
        public void SearchCallsServiceAndGeneratesProperId()
        {
            var item1 = new Activity
            {
                BingId = "NYX1111",
                Location = new double[] { 35.317366, -76.289062 },
                Name = "TestActivity1"
            };

            var expectedItem1Id = "NYX1111|1|35.317366~-76.289062";

            var item2 = new Activity
            {
                BingId = "NYX2222",
                Location = new double[] { 35.217366, -76.189062 },
                Name = "TestActivity2"
            };

            var expectedItem2Id = "NYX2222|1|35.217366~-76.189062";

            var expectedResult = new PagingResult<Activity>(new[] { item1, item2 });
            expectedResult.TotalItems = 2;
            expectedResult.PageSize = 5;
            expectedResult.CurrentPage = 1;

            var service = new Mock<IBingMapsService>();
            service.Setup(o => o.GetClientToken(It.IsAny<string>(), It.IsAny<int>()))
                   .Returns("ve-token");

            service.Setup(o => o.SearchActivities(It.IsAny<AdvancedSearchQuery>(), "ve-token"))
                   .Returns(expectedResult);

            var repo = new BingActivitiesRepository(service.Object);
            var result = repo.Search(new AdvancedSearchQuery
            {
                ActivityTypeId = 1,
                City = "New York",
                State = "NY",
                Page = 1,
                PageSize = 5
            });

            Assert.IsNotNull(result);
            Assert.AreEqual(expectedResult, result);
            Assert.AreEqual(ToBase64(expectedItem1Id), result.Items.Where(a => a.Name == "TestActivity1").Single().Id);
            Assert.AreEqual(ToBase64(expectedItem2Id), result.Items.Where(a => a.Name == "TestActivity2").Single().Id);

            service.VerifyAll();
        }

        [TestMethod]
        public void SearchUsingRadiusOrdersByDistanceAndFiltersLocally()
        {
            var item = new Activity
            {
                BingId = "NYX1111",
                Location = new double[] { 35.317366, -76.289062 },
                Name = "TestActivity1",
                Distance = 7.5
            };

            var itemOutOfScope = new Activity
            {
                BingId = "NYX3333",
                Location = new double[] { 35.117366, -76.089062 },
                Name = "TestActivityOutOfScope",
                Distance = 11.1
            };

            var expectedResult = new PagingResult<Activity>(new[] { item, item, item, itemOutOfScope, itemOutOfScope });
            expectedResult.TotalItems = 250;
            expectedResult.PageSize = 5;
            expectedResult.CurrentPage = 2;

            var service = new Mock<IBingMapsService>();
            service.Setup(o => o.GetClientToken(It.IsAny<string>(), It.IsAny<int>()))
                   .Returns("ve-token");

            service.Setup(o => o.SearchActivities(It.Is<AdvancedSearchQuery>(i => i.SortBy == SortCriteria.Distance && i.Radius == 10), "ve-token"))
                   .Returns(expectedResult);

            var repo = new BingActivitiesRepository(service.Object);
            var result = repo.Search(new AdvancedSearchQuery
            {
                ActivityTypeId = 1,
                City = "New York",
                State = "NY",
                Radius = 10,            // miles
                Page = 2,
                PageSize = 5
            });

            Assert.IsNotNull(result);
            Assert.AreEqual(3, result.Items.Count);
            Assert.AreEqual(8, result.TotalItems);
            Assert.AreEqual(2, result.TotalPages);
            Assert.AreEqual(0, result.Items.Where(a => a.Distance > 10).Count());

            service.VerifyAll();
        }

        [TestMethod]
        public void OpenSearchCallsServiceAndGeneratesProperId()
        {
            var item1 = new Activity
            {
                BingId = "NYX1111",
                Location = new double[] { 35.317366, -76.289062 },
                Name = "TestActivity1"
            };

            var expectedItem1Id = "NYX1111|1|35.317366~-76.289062";

            var item2 = new Activity
            {
                BingId = "NYX2222",
                Location = new double[] { 35.217366, -76.189062 },
                Name = "TestActivity2"
            };

            var expectedItem2Id = "NYX2222|1|35.217366~-76.189062";

            var expectedResult = new Tuple<PagingResult<Activity>, ActivityAddress>(
                new PagingResult<Activity>(new[] { item1, item2 }),
                new ActivityAddress { State = "NY", City = "New York" });

            expectedResult.Item1.TotalItems = 2;
            expectedResult.Item1.PageSize = 5;
            expectedResult.Item1.CurrentPage = 1;

            var service = new Mock<IBingMapsService>();
            service.Setup(o => o.GetClientToken(It.IsAny<string>(), It.IsAny<int>()))
                   .Returns("ve-token");

            service.Setup(o => o.SearchActivities(It.IsAny<NaturalSearchQuery>(), "ve-token"))
                   .Returns(expectedResult);

            var repo = new BingActivitiesRepository(service.Object);
            var result = repo.Search(new NaturalSearchQuery
            {
                ActivityTypeId = 1,
                Query = "Bars in Manhattan, NY",
                Page = 1,
                PageSize = 5
            });

            Assert.IsNotNull(result);
            Assert.AreEqual(expectedResult.Item1, result);
            Assert.AreEqual(ToBase64(expectedItem1Id), result.Items.Where(a => a.Name == "TestActivity1").Single().Id);
            Assert.AreEqual(ToBase64(expectedItem2Id), result.Items.Where(a => a.Name == "TestActivity2").Single().Id);
            service.VerifyAll();
        }

        [TestMethod]
        public void GeocodeAddressCallService()
        {
            var service = new Mock<IBingMapsService>();
            service.Setup(o => o.GetClientToken(It.IsAny<string>(), It.IsAny<int>()))
                   .Returns("ve-token");
            service.Setup(o => o.GeocodeAddress(It.IsAny<ActivityAddress>(), "ve-token"))
                   .Returns(new BingCoordinate { Latitude = 50, Longitude = 60 });

            var repo = new BingActivitiesRepository(service.Object);
            var result = repo.GeocodeAddress(new ActivityAddress { City = "Manhattan", State = "NY", StreetAddress = "123 Somewhere St" });

            Assert.IsNotNull(result);
            service.VerifyAll();
        }

        [TestMethod]
        public void ParseQueryLocationCallsService()
        {
            var service = new Mock<IBingMapsService>();
            service.Setup(o => o.GetClientToken(It.IsAny<string>(), It.IsAny<int>()))
                   .Returns("ve-token");
            service.Setup(o => o.SearchActivities(It.Is<NaturalSearchQuery>(s => s.Query == "bars in manhattan, ny"), "ve-token"))
                   .Returns(new Tuple<PagingResult<Activity>, ActivityAddress>(new PagingResult<Activity>(), new ActivityAddress { City = "Manhattan", State = "New York" }));

            var repo = new BingActivitiesRepository(service.Object);
            var result = repo.ParseQueryLocation("bars in manhattan, ny");

            Assert.IsNotNull(result);
            service.VerifyAll();
        }

        [TestMethod]
        public void RetrieveActivityCallsServiceAndGeneratesProperId()
        {
            var searchId = ToBase64("NYX2222|1|35.217366~-76.189062");

            var item1 = new Activity
            {
                BingId = "NYX1111",
                Location = new double[] { 35.317366, -76.289062 },
                Name = "TestActivity1"
            };

            var item2 = new Activity
            {
                BingId = "NYX2222",
                Location = new double[] { 35.217366, -76.189062 },
                Name = "TestActivity2"
            };

            var expectedResult = new PagingResult<Activity>(new[] { item1, item2 });
            expectedResult.TotalItems = 2;
            expectedResult.PageSize = 5;
            expectedResult.CurrentPage = 1;

            var service = new Mock<IBingMapsService>();
            service.Setup(o => o.GetClientToken(It.IsAny<string>(), It.IsAny<int>()))
                   .Returns("ve-token");

            service.Setup(o => o.SearchActivities(It.Is<AdvancedSearchQuery>(i => i.ActivityTypeId == 1 && i.Longitude == 35.217366 && i.Latitude == -76.189062 && i.SortBy == SortCriteria.Distance), "ve-token"))
                   .Returns(expectedResult);

            var repo = new BingActivitiesRepository(service.Object);
            var result = repo.RetrieveActivity(searchId);

            Assert.IsNotNull(result);
            Assert.AreEqual(item2, result);
            Assert.AreEqual("NYX2222", result.BingId);
            Assert.AreEqual(searchId, result.Id);
            Assert.AreEqual(1, result.TypeId);

            service.VerifyAll();
        }

        [TestMethod]
        public void RetrieveActivityUsingRetryPolicy()
        {
            var searchId = ToBase64("NYX2222|1|35.217366~-76.189062");

            var otherItem = new Activity
            {
                BingId = "NYX1111",
                Location = new double[] { 35.317366, -76.289062 },
                Name = "TestActivity1"
            };

            var expectedItem = new Activity
            {
                BingId = "NYX2222",
                Location = new double[] { 35.217366, -76.189062 },
                Name = "TestActivity2"
            };

            var group1 = new PagingResult<Activity>(new[] { otherItem, otherItem, otherItem, otherItem, otherItem, otherItem, otherItem, otherItem, otherItem, otherItem })
            {
                TotalItems = 250,
                PageSize = 10,
                CurrentPage = 1
            };

            var group2 = new PagingResult<Activity>(new[] { otherItem, otherItem, otherItem, otherItem, otherItem, otherItem, otherItem, otherItem, otherItem, otherItem })
            {
                TotalItems = 250,
                PageSize = 10,
                CurrentPage = 2
            };

            var group3 = new PagingResult<Activity>(new[] { otherItem, expectedItem, otherItem, otherItem, otherItem, otherItem, otherItem, otherItem, otherItem, otherItem })
            {
                TotalItems = 250,
                PageSize = 10,
                CurrentPage = 3
            };

            var service = new Mock<IBingMapsService>();
            service.Setup(o => o.GetClientToken(It.IsAny<string>(), It.IsAny<int>()))
                   .Returns("ve-token");

            service.Setup(o => o.SearchActivities(It.Is<AdvancedSearchQuery>(i => i.Page == 1 && i.PageSize == 10 && i.ActivityTypeId == 1 && i.Longitude == 35.217366 && i.Latitude == -76.189062 && i.SortBy == SortCriteria.Distance), "ve-token"))
                   .Returns(group1);
            service.Setup(o => o.SearchActivities(It.Is<AdvancedSearchQuery>(i => i.Page == 2 && i.PageSize == 10 && i.ActivityTypeId == 1 && i.Longitude == 35.217366 && i.Latitude == -76.189062 && i.SortBy == SortCriteria.Distance), "ve-token"))
                   .Returns(group2);
            service.Setup(o => o.SearchActivities(It.Is<AdvancedSearchQuery>(i => i.Page == 3 && i.PageSize == 10 && i.ActivityTypeId == 1 && i.Longitude == 35.217366 && i.Latitude == -76.189062 && i.SortBy == SortCriteria.Distance), "ve-token"))
                   .Returns(group3);

            var repo = new BingActivitiesRepository(service.Object);
            var result = repo.RetrieveActivity(searchId);

            Assert.IsNotNull(result);
            Assert.AreEqual(expectedItem, result);
            Assert.AreEqual("NYX2222", result.BingId);
            Assert.AreEqual(searchId, result.Id);

            service.VerifyAll();
        }

        [TestMethod]
        public void PopulateItineraryActivitiesCallServiceForEachItem()
        {
            var itinerary = new Itinerary();
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = ToBase64("NYX1111|1|35.317366~-76.289062") });
            itinerary.Activities.Add(new ItineraryActivity { ActivityId = ToBase64("NYX3333|2|35.117366~-76.089062") });

            var item1 = new Activity
            {
                BingId = "NYX1111",
                Location = new double[] { 35.317366, -76.289062 },
                Name = "TestActivity1"
            };

            var item2 = new Activity
            {
                BingId = "NYX3333",
                Location = new double[] { 35.117366, -76.089062 },
                Name = "TestActivity2"
            };

            var itemOutOfScope = new Activity
            {
                BingId = "NYX6666",
                Location = new double[] { 30.117366, -75.089062 },
                Name = "TestActivityOutOfScope",
                Distance = 50
            };

            var service = new Mock<IBingMapsService>();
            service.Setup(o => o.GetClientToken(It.IsAny<string>(), It.IsAny<int>()))
                   .Returns("ve-token");
            service.Setup(o => o.SearchActivities(It.Is<AdvancedSearchQuery>(s => s.Latitude == -76.289062 && s.Longitude == 35.317366 && s.ActivityTypeId == 1), It.IsAny<string>()))
                   .Returns(new PagingResult<Activity>(new[] { itemOutOfScope, item1, itemOutOfScope, item2, itemOutOfScope }));
            service.Setup(o => o.SearchActivities(It.Is<AdvancedSearchQuery>(s => s.Latitude == -76.089062 && s.Longitude == 35.117366 && s.ActivityTypeId == 2), It.IsAny<string>()))
                   .Returns(new PagingResult<Activity>(new[] { item2, itemOutOfScope, itemOutOfScope, itemOutOfScope, itemOutOfScope }));

            var repo = new BingActivitiesRepository(service.Object);
            repo.PopulateItineraryActivities(itinerary);

            Assert.IsNotNull(itinerary.Activities[0].Activity);
            Assert.IsNotNull(itinerary.Activities[1].Activity);
            Assert.AreEqual("TestActivity1", itinerary.Activities[0].Activity.Name);
            Assert.AreEqual("TestActivity2", itinerary.Activities[1].Activity.Name);

            service.VerifyAll();
        }

        [TestMethod]
        public void BingTokenExpired()
        {
            var token = new BingToken(TimeSpan.FromMinutes(-10), "bing-dummy-token");
            Assert.AreEqual("bing-dummy-token", token.Value);
            Assert.IsTrue(token.Expired);
        }

        [TestMethod]
        public void BingTokenValid()
        {
            var token = new BingToken(TimeSpan.FromMinutes(10), "bing-dummy-token");
            Assert.AreEqual("bing-dummy-token", token.Value);
            Assert.IsFalse(token.Expired);
        }

        ////// Cached
        ////[TestMethod]
        ////public void SearchCallsServiceGeneratesProperIdAndStoresInCache()
        ////{
        ////    var item1 = new Activity
        ////    {
        ////        BingId = "NYX1111",
        ////        Location = new double[] { 35.317366, -76.289062 },
        ////        Name = "TestActivity1"
        ////    };

        ////    var expectedItem1Id = "NYX1111|1|35.317366~-76.289062";

        ////    var item2 = new Activity
        ////    {
        ////        BingId = "NYX2222",
        ////        Location = new double[] { 35.217366, -76.189062 },
        ////        Name = "TestActivity2"
        ////    };

        ////    var expectedItem2Id = "NYX2222|1|35.217366~-76.189062";

        ////    var expectedResult = new PagingResult<Activity>(new[] { item1, item2 });
        ////    expectedResult.TotalItems = 2;
        ////    expectedResult.PageSize = 5;
        ////    expectedResult.CurrentPage = 1;

        ////    var service = new Mock<IBingMapsService>();
        ////    service.Setup(o => o.GetClientToken(It.IsAny<string>(), It.IsAny<int>()))
        ////           .Returns("ve-token");

        ////    service.Setup(o => o.SearchActivities(It.IsAny<AdvancedSearchQuery>(), "ve-token"))
        ////           .Returns(expectedResult);

        ////    var cache = new Mock<CacheBase>();

        ////    // token
        ////    cache.Setup(o => o.Get("BingToken")).Returns(null);
        ////    cache.Setup(o => o.Insert("BingToken", It.IsAny<string>(), It.IsAny<CacheDependency>(), It.IsAny<DateTime>(), Cache.NoSlidingExpiration, CacheItemPriority.NotRemovable, It.IsAny<CacheItemRemovedCallback>()));

        ////    // results
        ////    cache.Setup(o => o.Insert(It.Is<string>(s => s.StartsWith("ActivitySearch:[")), It.IsAny<PagingResult<Activity>>(), It.IsAny<CacheDependency>(), It.IsAny<DateTime>(), Cache.NoSlidingExpiration, CacheItemPriority.High, It.IsAny<CacheItemRemovedCallback>()));

        ////    // activities
        ////    cache.Setup(o => o.Get("Activity:NYX1111")).Returns(null);
        ////    cache.Setup(o => o.Insert("Activity:NYX1111", It.IsAny<Activity>(), It.IsAny<CacheDependency>(), It.IsAny<DateTime>(), Cache.NoSlidingExpiration, CacheItemPriority.Low, It.IsAny<CacheItemRemovedCallback>()));
        ////    cache.Setup(o => o.Get("Activity:NYX2222")).Returns(null);
        ////    cache.Setup(o => o.Insert("Activity:NYX2222", It.IsAny<Activity>(), It.IsAny<CacheDependency>(), It.IsAny<DateTime>(), Cache.NoSlidingExpiration, CacheItemPriority.Low, It.IsAny<CacheItemRemovedCallback>()));

        ////    var repo = new BingActivitiesRepository(service.Object, cache.Object);
        ////    var result = repo.Search(new AdvancedSearchQuery
        ////    {
        ////        ActivityTypeId = 1,
        ////        City = "New York",
        ////        State = "NY",
        ////        Page = 1,
        ////        PageSize = 5
        ////    });

        ////    Assert.IsNotNull(result);
        ////    Assert.AreEqual(expectedResult, result);
        ////    Assert.AreEqual(ToBase64(expectedItem1Id), result.Items.Where(a => a.Name == "TestActivity1").Single().Id);
        ////    Assert.AreEqual(ToBase64(expectedItem2Id), result.Items.Where(a => a.Name == "TestActivity2").Single().Id);

        ////    service.VerifyAll();
        ////    cache.VerifyAll();
        ////}

        ////[TestMethod]
        ////public void RetrieveActivityCallsServiceGeneratesProperIdAndStoresInCache()
        ////{
        ////    var searchId = ToBase64("NYX2222|1|35.217366~-76.189062");

        ////    var item1 = new Activity
        ////    {
        ////        BingId = "NYX1111",
        ////        Location = new double[] { 35.317366, -76.289062 },
        ////        Name = "TestActivity1"
        ////    };

        ////    var item2 = new Activity
        ////    {
        ////        BingId = "NYX2222",
        ////        Location = new double[] { 35.217366, -76.189062 },
        ////        Name = "TestActivity2"
        ////    };

        ////    var expectedResult = new PagingResult<Activity>(new[] { item1, item2 });
        ////    expectedResult.TotalItems = 2;
        ////    expectedResult.PageSize = 5;
        ////    expectedResult.CurrentPage = 1;

        ////    var service = new Mock<IBingMapsService>();
        ////    service.Setup(o => o.GetClientToken(It.IsAny<string>(), It.IsAny<int>()))
        ////           .Returns("ve-token");

        ////    var cache = new Mock<CacheBase>();
        ////    cache.Setup(o => o.Get("BingToken")).Returns(null);
        ////    cache.Setup(o => o.Insert("BingToken", It.IsAny<string>(), It.IsAny<CacheDependency>(), It.IsAny<DateTime>(), Cache.NoSlidingExpiration, CacheItemPriority.NotRemovable, It.IsAny<CacheItemRemovedCallback>()));
        ////    cache.Setup(o => o.Get("Activity:NYX2222")).Returns(null);
        ////    cache.Setup(o => o.Insert("Activity:NYX2222", It.IsAny<Activity>(), It.IsAny<CacheDependency>(), It.IsAny<DateTime>(), Cache.NoSlidingExpiration, CacheItemPriority.Normal, It.IsAny<CacheItemRemovedCallback>()));

        ////    service.Setup(o => o.SearchActivities(It.Is<AdvancedSearchQuery>(i => i.ActivityTypeId == 1 && i.Longitude == 35.217366 && i.Latitude == -76.189062 && i.SortBy == SortCriteria.Distance), "ve-token"))
        ////           .Returns(expectedResult);

        ////    var repo = new BingActivitiesRepository(service.Object, cache.Object);
        ////    var result = repo.RetrieveActivity(searchId);

        ////    Assert.IsNotNull(result);
        ////    Assert.AreEqual(item2, result);
        ////    Assert.AreEqual("NYX2222", result.BingId);
        ////    Assert.AreEqual(searchId, result.Id);

        ////    service.VerifyAll();
        ////    cache.VerifyAll();
        ////}

        ////// Cached
        ////[TestMethod]
        ////public void SearchFromCache()
        ////{
        ////    var item1 = new Activity
        ////    {
        ////        BingId = "NYX1111",
        ////        Location = new double[] { 35.317366, -76.289062 },
        ////        Name = "TestActivity1"
        ////    };

        ////    var item2 = new Activity
        ////    {
        ////        BingId = "NYX2222",
        ////        Location = new double[] { 35.217366, -76.189062 },
        ////        Name = "TestActivity2"
        ////    };

        ////    var expectedResult = new PagingResult<Activity>(new[] { item1, item2 });
        ////    expectedResult.TotalItems = 2;
        ////    expectedResult.PageSize = 5;
        ////    expectedResult.CurrentPage = 1;

        ////    var service = new Mock<IBingMapsService>();
        ////    var cache = new Mock<CacheBase>();

        ////    // token
        ////    cache.Setup(o => o.Get(It.Is<string>(s => s.StartsWith("ActivitySearch:["))))
        ////         .Returns(expectedResult);

        ////    var repo = new BingActivitiesRepository(service.Object, cache.Object);
        ////    var result = repo.Search(new AdvancedSearchQuery
        ////    {
        ////        ActivityTypeId = 1,
        ////        City = "New York",
        ////        State = "NY",
        ////        Page = 1,
        ////        PageSize = 5
        ////    });

        ////    Assert.IsNotNull(result);
        ////    Assert.AreEqual(expectedResult, result);

        ////    cache.VerifyAll();
        ////}

        ////[TestMethod]
        ////public void RetrieveActivityFromCache()
        ////{
        ////    var searchId = ToBase64("NYX2222|1|35.217366~-76.189062");

        ////    var activity = new Activity
        ////    {
        ////        BingId = "NYX2222",
        ////        Location = new double[] { 35.217366, -76.189062 },
        ////        Name = "TestActivity2"
        ////    };

        ////    var service = new Mock<IBingMapsService>();

        ////    var cache = new Mock<CacheBase>();
        ////    cache.Setup(o => o.Get("Activity:NYX2222")).Returns(activity);

        ////    var repo = new BingActivitiesRepository(service.Object, cache.Object);
        ////    var result = repo.RetrieveActivity(searchId);

        ////    Assert.IsNotNull(result);
        ////    Assert.AreEqual(activity, result);
        ////    Assert.AreEqual("NYX2222", result.BingId);

        ////    cache.VerifyAll();
        ////}

        private static string ToBase64(string key)
        {
            return Convert.ToBase64String(Encoding.Default.GetBytes(key));
        }
    }
}
